#!/bin/sh

set -eu

if [ $# != 2 ] ; then
    echo usage: $0 '<user>' '<hostname>'
    exit 1
fi

is_systemd () {
    case "$(readlink -f /proc/1/exe || true)" in
    *systemd*)
        return 0
        ;;
    *)
        return 1
        ;;
    esac
}

# the SUFFIX must not contain spaces or shell meta characters
SUFFIX=$1

if ! id -u $SUFFIX > /dev/null 2>&1; then
  adduser $SUFFIX --quiet --gecos "" --disabled-password --disabled-login
fi;

OFFSET=7000
PHPOFFSET=8000
USERID=`id -u $SUFFIX`
PORT=$((USERID+OFFSET))
PHPPORT=$((USERID+PHPOFFSET))
HOSTNAME=$2

if [ -e /etc/apache2-$SUFFIX ] ; then
    echo ERROR: /etc/apache2-$SUFFIX already exists
    exit 2
fi

mkdir -p /home/$SUFFIX/html
chown -R $SUFFIX:$SUFFIX /home/$SUFFIX
chmod -R 770 /home/$SUFFIX

echo Setting up /etc/apache2-$SUFFIX ...
cp -a /etc/apache2 /etc/apache2-$SUFFIX
rm -f /etc/apache2-$SUFFIX/sites-available/*proxy.conf
rm -f /etc/apache2-$SUFFIX/sites-enabled/*proxy.conf

mv /etc/apache2-$SUFFIX/apache2-base.conf /etc/apache2-$SUFFIX/apache2.conf
perl -p -i -e s,XXX,$SUFFIX, /etc/apache2-$SUFFIX/apache2.conf
perl -p -i -e s,80,$PORT, /etc/apache2-$SUFFIX/ports.conf
perl -p -i -e s,80,$PORT, /etc/apache2-$SUFFIX/sites-available/000-default.conf
perl -p -i -e s,/var/www/html,/home/$SUFFIX/html, /etc/apache2-$SUFFIX/sites-available/000-default.conf
perl -p -i -e s,fcgi://localhost,fcgi://localhost:$PHPPORT, /etc/apache2-$SUFFIX/conf-available/php7.2-fpm.conf

if is_systemd ; then
    echo "systemd is in use, no init script installed"
    echo "use the 'apache2@$SUFFIX.service' service to control your new instance"
    echo "sample commands:"
    echo "systemctl start apache2@$SUFFIX.service"
    echo "systemctl enable apache2@$SUFFIX.service"
else
    echo "Setting up /etc/init.d/apache2-$SUFFIX ..."
    cp /usr/share/doc/apache2/examples/secondary-init-script /etc/init.d/apache2-$SUFFIX
    # adjust service name (this prevents us from using a simple symlink)
    perl -p -i -e s,XXX,$SUFFIX, /etc/init.d/apache2-$SUFFIX
    chmod 755 /etc/init.d/apache2-$SUFFIX
fi

echo -n Setting up symlinks:
for a in a2enmod a2dismod a2ensite a2dissite a2enconf a2disconf apache2ctl ; do
    echo -n " $a-$SUFFIX"
    ln -s /usr/sbin/$a /usr/local/sbin/$a-$SUFFIX
done
echo

echo Setting up /etc/logrotate.d/apache2-$SUFFIX and /var/log/apache2-$SUFFIX ...
cp -a /etc/logrotate.d/apache2 /etc/logrotate.d/apache2-$SUFFIX
perl -p -i -e s,apache2,apache2-$SUFFIX,g /etc/logrotate.d/apache2-$SUFFIX
mkdir /var/log/apache2-$SUFFIX
chmod 750 /var/log/apache2-$SUFFIX
chown root:adm /var/log/apache2-$SUFFIX

echo "Setting up /etc/default/apache-htcacheclean-$SUFFIX"
cp -a /etc/default/apache-htcacheclean /etc/default/apache-htcacheclean-$SUFFIX

echo "<VirtualHost *:80>
  ServerName $HOSTNAME
  ProxyPass / http://localhost:$PORT/
  ProxyPassReverse / http://localhost:$PORT/
</VirtualHost>" > /etc/apache2/sites-available/$SUFFIX-proxy.conf


echo "[$SUFFIX]
user = $SUFFIX
group = $SUFFIX
listen = 127.0.0.1:$PHPPORT
listen.allowed_clients = 127.0.0.1
pm = ondemand
pm.max_children = 4
php_admin_value[error_log] = /var/log/apache2-$SUFFIX/php-fpm-error.log" > /etc/php/7.2/fpm/pool.d/$SUFFIX.conf

a2ensite $SUFFIX-proxy
systemctl reload apache2.service
systemctl enable apache2@$SUFFIX.service
service php7.2-fpm reload
systemctl start apache2@$SUFFIX.service
